/**
 * \file: util_hash.c
 * 
 * \version: $Revision: 1.11 $
 * 
 * \release: $Name: UTIL_r2_E05 $
 *
 * Implementation of a hash table with chaining.
 * 
 * \component: Utility
 * 
 * \author T. Polle / ADIT / SWG / tpolle(o)de.adit-jv.com
 * 
 * \copyright: (c) 2003 - 2008 ADIT Corporation 
 */

#include <stdlib.h>
#include <string.h>

#include "util_lib.h"
#include "util_hash_i.h"

#define UTIL_HASH_NAME "ADUTILh"

#define UTIL_HASH_03    3
#define UTIL_HASH_06    6
#define UTIL_HASH_10   10
#define UTIL_HASH_11   11
#define UTIL_HASH_15   15

LOCAL U32 UTIL_hash_func(U32 nel, UTIL_hash_elem_t* elem);
LOCAL U32 UTIL_hash_func(U32 nel, UTIL_hash_elem_t* elem)
{
  U8* key  = elem->key;
  U32  hash = 0;
  U32  i    = 0;
 
  elem = elem;

  for (i = 0; i < elem->key_len; i++)
  {
    hash += key[i];
    hash += (hash << UTIL_HASH_10);
    hash ^= (hash >> UTIL_HASH_06);
  }
  hash += (hash << UTIL_HASH_03);
  hash ^= (hash >> UTIL_HASH_11);
  hash += (hash << UTIL_HASH_15);
  
  if (0 == nel)
  {
      hash = 0;
  }
  else
  {
      hash = hash % nel;
  }
  return hash;
}

LOCAL void UTIL_init_table(UTIL_hash_table_t*     table,
                           U32                     nel,
                           const UTIL_hash_cmp_fn func,
                           UTIL_mpf_t*            bucket_pool);
LOCAL void UTIL_init_table(UTIL_hash_table_t*     table,
                           U32                     nel,
                           const UTIL_hash_cmp_fn func,
                           UTIL_mpf_t*            bucket_pool)
{
  U32 size    = (sizeof(*table) + (nel * sizeof(UTIL_hash_bucket_t*)));
  VP buckets = ((U8*)(VP)table + sizeof(*table));

  memset(table, 0, size);
  table->bucket_pool = bucket_pool;
  table->nel         = nel;
  table->func        = func;
  table->buckets     = buckets;
}

/* PRQA: QAC Message 5118: Heap allocation necessary. */
/* PRQA S 5118 L1 */
EXPORT UTIL_hash_table_t* UTIL_cre_hsh(U32 nel, const UTIL_hash_cmp_fn func)
{
  UTIL_hash_table_t* table = NULL;
  U32                 size  = (sizeof(*table)
                              + (nel * sizeof(UTIL_hash_bucket_t*)));
  table = malloc(size);
  if (NULL != table)
  {
    UTIL_CMPF_t cmpf = {NULL, TA_DSNAME, (S32)nel, sizeof(UTIL_hash_bucket_t),
                        UTIL_HASH_NAME, NULL};
    table->bucket_pool = UTIL_cre_mpf(&cmpf);
    if (NULL != table->bucket_pool)
    {
      UTIL_init_table(table, nel, func, table->bucket_pool);
    }
    else
    {
      free(table);
      table = NULL;
    }
  }
  return table;
}

EXPORT void UTIL_clr_hsh(UTIL_hash_table_t* table)
{
  if (NULL != table)
  {
    UTIL_init_table(table, table->nel, table->func, table->bucket_pool);
    UTIL_init_mpf(table->bucket_pool);
  }
}

EXPORT S32 UTIL_del_hsh(UTIL_hash_table_t* table)
{
    table = table;

  if (NULL != table)
  {
    (void)UTIL_del_mpf(table->bucket_pool);
    free(table);
  }
  return E_OK;
}
/* PRQA L:L1 */

EXPORT UTIL_hash_elem_t* UTIL_sea_hsh(UTIL_hash_table_t* table,
                                      UTIL_hash_elem_t*  elem)
{
  U32                  hash   = UTIL_hash_func(table->nel, elem);
  UTIL_hash_bucket_t* bucket = table->buckets[hash];

  table = table;

  /* PRQA: QAC Message 3415: table->func is a comparison function
               without side effects. */
  /* PRQA S 3415 L1 */
  while ((NULL != bucket)
         &&
         (FALSE == table->func(elem, &bucket->elem)))
  /* PRQA L:L1 */
  {
    bucket = bucket->next;
  }
  if (NULL != bucket)
  {
    elem = &bucket->elem;
  }
  else
  {
    elem = NULL;
  }
  return elem;
}

EXPORT UTIL_hash_elem_t* UTIL_add_hsh(UTIL_hash_table_t* table,
                                      UTIL_hash_elem_t*  elem)
{
  U32                   hash   = UTIL_hash_func(table->nel, elem);
  UTIL_hash_bucket_t** chain  = &table->buckets[hash];
  UTIL_hash_bucket_t*  bucket = NULL;
  S32                   rc     = E_OK;

  table = table;

  /* Search for the end of the bucket chain. */
  while (NULL != (*chain))
  {
    chain = &(*chain)->next;
  }
  /* Create a new bucket and insert it to the chain. */
  rc = UTIL_get_mpf(table->bucket_pool, (VP*)(VP)&bucket, TMO_FEVR);
  if (E_OK == rc)
  {
    bucket->next = NULL;
    bucket->elem = (*elem);
    (*chain)     = bucket;
  }
  else
  {
    elem = NULL;
  }
  return elem;
}

EXPORT S32 UTIL_rem_hsh(UTIL_hash_table_t* table,
                       UTIL_hash_elem_t*  elem)
{
  U32                   hash  = UTIL_hash_func(table->nel, elem);
  UTIL_hash_bucket_t** chain = &table->buckets[hash];
  S32                   rc    = E_OK;

  table = table;
  hash  = hash;

  /* Search the elements in the chain for the element to be removed. */
  /* PRQA: QAC Message 3415: table->func is a comparison function
               without side effects. */
  /* PRQA S 3415 L1 */
  while ((NULL != (*chain)) && (FALSE == table->func(elem, &(*chain)->elem)))
  /* PRQA L:L1 */
  {
    chain = &(*chain)->next;
  }
  if (NULL != (*chain))
  {
    /* Remove the bucket. */
    UTIL_hash_bucket_t* rem = (*chain);
    (*chain) = (*chain)->next;
    rc = UTIL_rel_mpf(table->bucket_pool, rem);
  }
  return rc;
}
